/*
* This file is a part of the Cairo-Dock project
*
* Copyright : (C) see the 'copyright' file.
* E-mail    : see the 'copyright' file.
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 3
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#ifndef __CAIRO_DOCK_THEMES_MANAGER__
#define  __CAIRO_DOCK_THEMES_MANAGER__

#include <glib.h>
#include <gtk/gtk.h>
G_BEGIN_DECLS

/**@file cairo-dock-themes-manager.h This class defines the structure of the global theme (launchers, icons, plug-ins, configuration files, etc).
* It also provides methods to manage the themes, like exporting the current theme, importing new themes, deleting themes, etc.
*/

gboolean cairo_dock_current_theme_need_save (void);

void cairo_dock_delete_conf_file (const gchar *cConfFilePath);

gboolean cairo_dock_add_conf_file (const gchar *cOriginalConfFilePath, const gchar *cConfFilePath);

/** Update a conf file with a list of values of the form : {type, name of the groupe, name of the key, value}. Must end with G_TYPE_INVALID.
*@param cConfFilePath path to the conf file.
*@param iFirstDataType type of the first value.
*/
void cairo_dock_update_conf_file (const gchar *cConfFilePath, GType iFirstDataType, ...);

/** Write a key file on the disk.
*@param pKeyFile the key-file
*@param cConfFilePath its path on the disk
*/
void cairo_dock_write_keys_to_conf_file (GKeyFile *pKeyFile, const gchar *cConfFilePath);

/** Write a key file on the disk to a newly created configuration file.
*@param pKeyFile the key-file
*@param cConfFilePath its path on the disk
*@return The name of the new config file in a newly allocated string, or NULL on failure.
* If cConfFilePath does not exist, a new file is created using this path.
* If cConfFilePath already exists, a new filename is generated by using it as a template and adding a unique suffix, and the keyfile is written there instead.
*/
gchar *cairo_dock_write_keys_to_new_conf_file (GKeyFile *pKeyFile, const gchar *cConfFilePath);

/** Export the current theme to a given name. Exported themes can be imported directly from the Theme Manager.
 * @param cNewThemeName name to export the theme to.
 * @param bSaveBehavior whether to save the behavior parameters too.
 * @param bSaveLaunchers whether to save the launchers too.
 * @return TRUE if the theme could be exported succefuly.
 */
gboolean cairo_dock_export_current_theme (const gchar *cNewThemeName, gboolean bSaveBehavior, gboolean bSaveLaunchers);

/** Create a package of the current theme. Packages can be distributed easily, and imported into the dock by a mere drag and drop into the Theme Manager. The package is placed in the cDirPath directory (or $HOME if cDirPath is wrong).
 * @param cThemeName name of the package.
 * @param cDirPath path to the directory
 * @return TRUE if the theme could be packaged succefuly.
 */
gboolean cairo_dock_package_current_theme (const gchar *cThemeName, const gchar *cDirPath);


/** Extract a package into the themes folder. Does not load it.
 * @param cPackagePath path of a package. If the package is distant, it is first downoladed.
 * @return the path of the theme folder, or NULL if anerror occured.
 */
gchar * cairo_dock_depackage_theme (const gchar *cPackagePath);

/** Remove some exported themes from the hard-disk.
 * @param cThemesList a list of theme names, NULL-terminated.
 * @return TRUE if the themes has been succefuly deleted.
 */
gboolean cairo_dock_delete_themes (gchar **cThemesList);

/** Import a theme, which can be : a local theme, a user theme, a distant theme, or even the path to a packaged theme.
 * @param cThemeName name of the theme to import.
 * @param bLoadBehavior whether to import the behavior parameters too.
 * @param bLoadLaunchers whether to import the launchers too.
 * @return TRUE if the theme could be imported succefuly.
 */
gboolean cairo_dock_import_theme (const gchar *cThemeName, gboolean bLoadBehavior, gboolean bLoadLaunchers);

typedef void (*CairoDockImportThemeCB) (gboolean, gpointer);
/** Asynchronously import a theme, which can be : a local theme, a user theme, a distant theme, or even the path to a packaged theme. This function is non-blocking, you'll get a CairoTask that you can discard at any time, and you'll get the result of the import as the first argument of the callback (the second being the data you passed to this function).
 * Note that only downloading or unpacking the theme is done asynchronously, actually copying the files in the current theme folder is not (because it couldn't be cancelled without first making a backup).
 * @param cThemeName name of the theme to import.
 * @param bLoadBehavior whether to import the behavior parameters too.
 * @param bLoadLaunchers whether to import the launchers too.
 * @param pCallback function called when the download is finished. It takes the result of the import (TRUE for a successful import) and the data you've set here.
 * @param data data to be passed to the callback.
 * @return the Task that is doing the job. Keep it and use \ref cairo_dock_discard_task if you want to discard the download before it's completed (for instance if the user cancels it), or \ref cairo_dock_free_task inside your callback.
 */
GldiTask *cairo_dock_import_theme_async (const gchar *cThemeName, gboolean bLoadBehavior, gboolean bLoadLaunchers, CairoDockImportThemeCB pCallback, gpointer data);

/** Define the paths of themes. Do it just after 'gldi_init'.
*@param cRootDataDirPath path to the root folder of libgldi
*@param cExtraDirPath path to the extras themes (plug-in themes)
*@param cThemesDirPath path to the user themes
*@param cCurrentThemeDirPath path to the current theme
*@param cLocalThemeDirPath path to the installed themes (default themes)
*@param cDistantThemeDirName folder of the themes on the server
*@param cThemeServerAdress adress of the themes server
*/
void cairo_dock_set_paths (gchar *cRootDataDirPath, gchar *cExtraDirPath, gchar *cThemesDirPath, gchar *cCurrentThemeDirPath, gchar *cLocalThemeDirPath, gchar *cDistantThemeDirName, gchar *cThemeServerAdress);


G_END_DECLS
#endif
