"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.callTokenEndpoint = callTokenEndpoint;
exports.composeLogoutUrl = composeLogoutUrl;
exports.getBaseRedirectUrl = getBaseRedirectUrl;
exports.getExpirationDate = getExpirationDate;
exports.getNextUrl = getNextUrl;
exports.getRootUrl = getRootUrl;
exports.includeAdditionalParameters = includeAdditionalParameters;
exports.parseTokenResponse = parseTokenResponse;
var _querystring = require("querystring");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

function parseTokenResponse(payload) {
  const payloadString = payload.toString();
  if (payloadString.trim()[0] === '{') {
    try {
      return JSON.parse(payloadString);
    } catch (error) {
      throw Error(`Invalid JSON payload: ${error}`);
    }
  }
  return (0, _querystring.parse)(payloadString);
}
function getRootUrl(config, core, request) {
  var _config$openid;
  const host = core.http.getServerInfo().hostname;
  const port = core.http.getServerInfo().port;
  let protocol = core.http.getServerInfo().protocol;
  let httpHost = `${host}:${port}`;
  if ((_config$openid = config.openid) !== null && _config$openid !== void 0 && _config$openid.trust_dynamic_headers) {
    const xForwardedHost = request.headers['x-forwarded-host'] || undefined;
    const xForwardedProto = request.headers['x-forwarded-proto'] || undefined;
    if (xForwardedHost) {
      httpHost = xForwardedHost;
    }
    if (xForwardedProto) {
      protocol = xForwardedProto;
    }
  }
  return `${protocol}://${httpHost}`;
}
function getBaseRedirectUrl(config, core, request) {
  var _config$openid2;
  if ((_config$openid2 = config.openid) !== null && _config$openid2 !== void 0 && _config$openid2.base_redirect_url) {
    const baseRedirectUrl = config.openid.base_redirect_url;
    return baseRedirectUrl.endsWith('/') ? baseRedirectUrl.slice(0, -1) : baseRedirectUrl;
  }
  const rootUrl = getRootUrl(config, core, request);
  if (core.http.basePath.serverBasePath) {
    return `${rootUrl}${core.http.basePath.serverBasePath}`;
  }
  return rootUrl;
}
function getNextUrl(config, core, request) {
  return request.query.nextUrl || getBaseRedirectUrl(config, core, request) || '/';
}
async function callTokenEndpoint(tokenEndpoint, query, wreckClient) {
  var _tokenResponse$res;
  const tokenResponse = await wreckClient.post(tokenEndpoint, {
    payload: (0, _querystring.stringify)(query),
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded'
    }
  });
  if (!((_tokenResponse$res = tokenResponse.res) !== null && _tokenResponse$res !== void 0 && _tokenResponse$res.statusCode) || tokenResponse.res.statusCode < 200 || tokenResponse.res.statusCode > 299) {
    throw new Error(`Failed calling token endpoint: ${tokenResponse.res.statusCode} ${tokenResponse.res.statusMessage}`);
  }
  const tokenPayload = parseTokenResponse(tokenResponse.payload);
  return {
    idToken: tokenPayload.id_token,
    accessToken: tokenPayload.access_token,
    refreshToken: tokenPayload.refresh_token,
    expiresIn: tokenPayload.expires_in
  };
}
function composeLogoutUrl(customLogoutUrl, idpEndsessionEndpoint, additionalQueryParams) {
  const logoutEndpoint = customLogoutUrl || idpEndsessionEndpoint;
  const logoutUrl = new URL(logoutEndpoint);
  Object.keys(additionalQueryParams).forEach(key => {
    logoutUrl.searchParams.append(key, additionalQueryParams[key]);
  });
  return logoutUrl.toString();
}
function getExpirationDate(tokenResponse) {
  if (!tokenResponse) {
    throw new Error('Invalid token');
  } else if (tokenResponse.idToken) {
    const idToken = tokenResponse.idToken;
    const parts = idToken.split('.');
    if (parts.length !== 3) {
      throw new Error('Invalid token');
    }
    const claim = JSON.parse(Buffer.from(parts[1], 'base64').toString());
    return claim.exp * 1000;
  } else {
    return Date.now() + tokenResponse.expiresIn * 1000;
  }
}
function includeAdditionalParameters(query, context, config) {
  var _config$openid3;
  if ((_config$openid3 = config.openid) !== null && _config$openid3 !== void 0 && _config$openid3.additional_parameters) {
    for (const [key, value] of Object.entries((_config$openid4 = config.openid) === null || _config$openid4 === void 0 ? void 0 : _config$openid4.additional_parameters)) {
      var _config$openid4;
      if (query[key] == null) {
        query[key] = value;
      } else {
        context.security_plugin.logger.warn(`Additional parameter in OpenID config '${key}' was ignored as it would overwrite existing parameters`);
      }
    }
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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